/*
 * NASPRO - The NASPRO Architecture for Sound PROcessing
 * Portable runtime library
 *
 * Copyright (C) 2007-2014 Stefano D'Angelo
 *
 * See the COPYING file for license conditions.
 */

#define _WIN32_WINNT	0x501
#define UNICODE

#include "internal.h"

#include <windows.h>

struct _nacore_fs_dir
  {
	HANDLE			handle;
	char			first;
	WIN32_FIND_DATAW	fdata;
  };

_NACORE_DEF const char * const nacore_fs_dir_sep = "\\";

_NACORE_DEF nacore_fs_dir
nacore_fs_dir_open(const char *name)
{
	nacore_fs_dir ret;
	uint16_t *namew;
	char *tmp;
	int errsv;

	ret = malloc(sizeof(struct _nacore_fs_dir));
	if (ret == NULL)
	  {
		errsv = ENOMEM;
		goto alloc_err;
	  }

	nacore_asprintf_nl(&tmp, "%s\\*", name);
	if (tmp == NULL)
	  {
		errno = ENOMEM;
		free(ret);
		return NULL;
	  }

	namew = nacore_string_utf8_to_utf16_a(tmp);
	free(tmp);
	if (namew == NULL)
	  {
		errsv = ENOMEM;
		goto namew_err;
	  }

	ret->handle = FindFirstFileW((LPCWSTR)namew, &ret->fdata);
	free(namew);
	if (ret->handle == INVALID_HANDLE_VALUE)
	  {
		switch (GetLastError())
		  {
			/* error hopefully catched, just guessing... */
			case ERROR_ACCESS_DENIED:
			case ERROR_SHARING_VIOLATION:
			case ERROR_LOCK_VIOLATION:
			case ERROR_NOT_DOS_DISK:
			case ERROR_NETWORK_ACCESS_DENIED:
			case 0x326: // ERROR_FILE_HANDLE_REVOKED
			case ERROR_NOACCESS:
			case ERROR_ACCESS_DISABLED_BY_POLICY:
				errsv = EACCES;
				break;
			case 0x2a9: // ERROR_STOPPED_ON_SYMLINK
				errsv = ELOOP;
				break;
			case ERROR_TOO_MANY_OPEN_FILES:
			case ERROR_SHARING_BUFFER_EXCEEDED:
				errsv = ENFILE;
				break;
			case ERROR_BUFFER_OVERFLOW:
			case ERROR_FILENAME_EXCED_RANGE:
				errsv = ENAMETOOLONG;
				break;
			case ERROR_NOT_ENOUGH_MEMORY:
				errsv = ENOMEM;
				break;
			case ERROR_PATH_NOT_FOUND:
				errsv = ENOENT;
				break;
			case ERROR_FILE_NOT_FOUND:
				errsv = ENOTDIR;
				break;
			default:
				errsv = NACORE_EUNKNOWN;
				break;
		  }
		goto ff_err;
	  }

	ret->first = 1;

	return ret;

ff_err:
namew_err:
	free(ret);
alloc_err:
	errno = errsv;
	return NULL;
}

_NACORE_DEF nacore_fs_dir_entry
nacore_fs_dir_get_next_entry(nacore_fs_dir dir)
{
	char *ret;

	ret = NULL;
	do
	  {
		if (ret != NULL)
			free(ret);

		if (!dir->first)
		  {
			if (FindNextFileW(dir->handle, &dir->fdata) == 0)
			  {
				if (GetLastError() != ERROR_NO_MORE_FILES)
					errno = NACORE_EUNKNOWN;
				return NULL;
			  }
		  }

		dir->first = 0;

		ret = nacore_string_utf16_to_utf8_a(
				(const uint16_t *)dir->fdata.cFileName);
		if (ret == NULL)
		  {
			errno = ENOMEM;
			return NULL;
		  }
	  }
	while ((strcmp(ret, ".") == 0) || (strcmp(ret, "..") == 0));

	return (nacore_fs_dir_entry)ret;
}

_NACORE_DEF int
nacore_fs_dir_close(nacore_fs_dir dir)
{
	if (FindClose(dir->handle) == 0)
		return NACORE_EUNKNOWN;

	free(dir);

	return 0;
}

_NACORE_DEF const char *
nacore_fs_dir_entry_get_name(nacore_fs_dir_entry entry)
{
	return (const char *)entry;
}

_NACORE_DEF void
nacore_fs_dir_entry_free(nacore_fs_dir_entry entry)
{
	free(entry);
}
