(* Copyright (C) 2009,2019 Matthew Fluet.
 * Copyright (C) 1999-2006 Henry Cejtin, Matthew Fluet, Suresh
 *    Jagannathan, and Stephen Weeks.
 * Copyright (C) 1997-2000 NEC Research Institute.
 *
 * MLton is released under a HPND-style license.
 * See the file MLton-LICENSE for details.
 *)

(* SML/NJ uses an old version of datatype IEEEReal.float_class. *)
signature REAL =
   sig
      type real

      structure Math: MATH where type real = real

      val != : real * real -> bool
      val * : real * real -> real
      val *+ : real * real * real -> real
      val *- : real * real * real -> real
      val + : real * real -> real
      val - : real * real -> real
      val / : real * real -> real
      val <  : real * real -> bool
      val <= : real * real -> bool
      val == : real * real -> bool
      val >  : real * real -> bool
      val >= : real * real -> bool
      val ?= : real * real -> bool
      val abs: real -> real
      val checkFloat: real -> real
      val class: real -> IEEEReal.float_class
      val compare: real * real -> order
      val compareReal: real * real -> IEEEReal.real_order
      val copySign: real * real -> real
      val fmt: StringCvt.realfmt -> real -> string
      val fromDecimal: IEEEReal.decimal_approx -> real option
      val fromInt: int -> real
      val fromLarge: IEEEReal.rounding_mode -> LargeReal.real -> real
      val fromLargeInt: LargeInt.int -> real
      val fromManExp: {man: real, exp: int} -> real
      val fromString: string -> real option
      val isFinite: real -> bool
      val isNan: real -> bool
      val isNormal: real -> bool
      val max: real * real -> real
      val maxFinite: real
      val min: real * real -> real
      val minNormalPos: real
      val minPos: real
      val negInf: real
      val nextAfter: real * real -> real
      val posInf: real
      val precision: int
      val radix: int
      val realCeil: real -> real
      val realFloor: real -> real
      val realMod: real -> real
      val realRound: real -> real
      val realTrunc: real -> real
      val rem: real * real -> real
      val round: real -> Int.int
      val sameSign: real * real -> bool
      val scan: (char, 'a) StringCvt.reader -> (real, 'a) StringCvt.reader
      val sign: real -> int
      val signBit: real -> bool
      val split: real -> {whole: real, frac: real}
      val toDecimal: real -> IEEEReal.decimal_approx
      val toInt: IEEEReal.rounding_mode -> real -> int
      val toLarge: real -> LargeReal.real
      val toLargeInt: IEEEReal.rounding_mode -> real -> LargeInt.int
      val toManExp: real -> {man: real, exp: int}
      val toString: real -> string
      val unordered: real * real -> bool
      val ~ : real -> real
      val ceil: real -> Int.int
      val floor: real -> Int.int
      val trunc: real -> Int.int
   end

functor FixReal(PReal: sig include PERVASIVE_REAL val zero : real end) : REAL =
   struct
      open PReal

      (* SML/NJ uses an old version of datatype IEEEReal.float_class. *)
      local
         datatype z = datatype IEEEReal.float_class
         structure P = Pervasive.IEEEReal
         fun toGoodFC c =
            case c of
               P.NAN _ => NAN
             | P.INF => INF
             | P.ZERO => ZERO
             | P.NORMAL => NORMAL
             | P.SUBNORMAL => SUBNORMAL
         fun toBadFC c =
            case c of
               NAN => P.NAN P.QUIET
             | INF => P.INF
             | ZERO => P.ZERO
             | NORMAL => P.NORMAL
             | SUBNORMAL => P.SUBNORMAL
         fun toGoodDA {digits, exp, kind, sign} =
            {digits = digits, exp = exp, kind = toGoodFC kind, sign = sign}
         fun toBadDA {digits, exp, kind, sign} =
            {digits = digits, exp = exp, kind = toBadFC kind, sign = sign}
      in
         val class = toGoodFC o class
         val fromDecimal = SOME o fromDecimal o toBadDA
         val toDecimal = toGoodDA o toDecimal
      end

      (* SML/NJ doesn't support EXACT. *)
      fun fmt f =
         PReal.fmt
         (let
             datatype z = datatype StringCvt.realfmt
          in
             case f of
                EXACT => StringCvt.GEN NONE
              | FIX io => StringCvt.FIX io
              | GEN io => StringCvt.GEN io
              | SCI io => StringCvt.SCI io
          end)

      val fromString = PReal.fromString
      (* SML/NJ raises Overflow on large exponents. *)
      (* Fixed in SML/NJ 110.83. *)
      val fromString = fn s =>
         (case SOME (fromString s) handle Overflow => NONE of
             NONE =>
                let
                   val manexp =
                      String.tokens
                      (fn c => c = #"e" orelse c = #"E")
                      s
                   fun isNeg s =
                      String.sub (s, 0) = #"~"
                      orelse String.sub (s, 0) = #"+"
                   fun isNonzero s =
                      CharVector.exists
                      (fn c => Char.<= (#"1", c) andalso Char.<= (c, #"9"))
                      s
                in
                   case manexp of
                      [man,exp] =>
                         if isNeg exp
                            then SOME zero
                         else if isNonzero man
                            then SOME posInf
                         else SOME zero
                     | _ => NONE
                end
           | SOME ro => ro)
      (* SML/NJ doesn't handle "[+~-]?(inf|infinity|nan)". *)
      val fromString = fn s =>
         case s of
            "inf" => SOME posInf
          | "infinity" => SOME posInf
          | "+inf" => SOME posInf
          | "+infinity" => SOME posInf
          | "~inf" => SOME negInf
          | "~infinity" => SOME negInf
          | "-inf" => SOME negInf
          | "-infinity" => SOME negInf
          | "nan" => SOME (negInf + posInf)
          | "+nan" => SOME (negInf + posInf)
          | "~nan" => SOME (negInf + posInf)
          | "-nan" => SOME (negInf + posInf)
          | _ => fromString s
   end

structure LargeReal = FixReal(struct open Pervasive.LargeReal val zero : real = 0.0 end)
structure Real = FixReal(struct open Pervasive.Real val zero : real = 0.0 end)
structure Real64 = FixReal(struct open Pervasive.Real64 val zero : real = 0.0 end)
structure Real32 = Real64
