/*
 *  $Id: outliers.c 28910 2025-11-24 18:11:09Z yeti-dn $
 *  Copyright (C) 2004,2008 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gwy.h>
#include "preview.h"

#define RUN_MODES GWY_RUN_IMMEDIATE

static gboolean module_register(void);
static void     module_main    (GwyFile *data,
                                GwyRunModeFlags mode);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Creates mask of outliers."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.5",
    "David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, outliers)

static gboolean
module_register(void)
{
    gwy_process_func_register("outliers",
                              module_main,
                              N_("/_Correct Data/Mask of _Outliers"),
                              GWY_ICON_MARK_OUTLIERS,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE,
                              N_("Mark data farther than 3σ from mean value"));

    return TRUE;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    GwyField *dfield, *maskfield;
    GQuark dquark, mquark;
    gdouble thresh = 3.0;
    gboolean has_mask;
    gint xres, yres, count, id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD_KEY, &dquark,
                                 GWY_APP_FIELD, &dfield,
                                 GWY_APP_MASK_FIELD_KEY, &mquark,
                                 GWY_APP_MASK_FIELD, &maskfield,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(dfield && dquark);
    has_mask = !!maskfield;
    maskfield = gwy_field_new_alike(dfield, TRUE);
    gwy_unit_clear(gwy_field_get_unit_z(maskfield));

    gwy_field_mask_outliers(dfield, maskfield, thresh);

    /* Do not create useless undo levels if there was no mask and the new mask would be empty.  And do not create
     * empty masks at all, just remove the mask instead.  */
    xres = gwy_field_get_xres(maskfield);
    yres = gwy_field_get_yres(maskfield);
    gwy_field_area_count_in_range(maskfield, NULL, GWY_MASK_IGNORE, 0, 0, xres, yres, 0.0, 0.0, &count, NULL);
    count = xres*yres - count;

    if (count || has_mask) {
        gwy_app_undo_qcheckpointv(GWY_CONTAINER(data), 1, &mquark);

        if (count) {
            gwy_file_pass_image_mask(data, id, maskfield);
            maskfield = NULL;
        }
        else
            gwy_container_remove(GWY_CONTAINER(data), mquark);

        gwy_log_add(data, GWY_FILE_IMAGE, id, id);
    }
    g_clear_object(&maskfield);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
