/* PROGRAM:	eggsh
 * FILE:	$Header: /home/egg/src/RCS/reg_pear.c,v 1.1 1998/12/31 22:04:39 ghn Exp $
 * PURPOSE:	PEAR (Bradish box/micro-REG) hardware interface
 * AUTHOR:	Greg Nelson
 * DATE:	98-04-12
 *
 * REVISED:	$Log: reg_pear.c,v $
 * REVISED:	Revision 1.1  1998/12/31 22:04:39  ghn
 * REVISED:	Initial revision
 * REVISED:
 * REVISED:	Revision 1.3  1998/08/01 18:50:39  ghn
 * REVISED:     Added John's byte-order-independence changes and PSEUDO support.
 * REVISED:
 * REVISED:	Revision 1.2  1998/08/01 17:13:51  ghn
 * REVISED:     Added John's Solaris support and DUMPREG option.
 * REVISED:
 * REVISED:	Revision 1.1  1998/07/21 11:37:41  ghn
 * REVISED:	Initial revision
 * REVISED:
 * Copyright 1998 - Greg Nelson
 * Redistributable under the terms of the GNU Public Licence (GPL)
 */

/* Define this to dump samples from the REG which we actually use into
   a file named dumpreg.dat.  This can be used to debug serial port
   mode problems (for example, setting the character frame to 7 bits,
   or telling it to ignore breaks, which will lose all zero bytes. */
/* #define DUMPREG */

#include <stdio.h>
#ifdef __USE_BSD
#undef __USE_BSD
#endif
#define __USE_BSD
#include <termios.h>
#undef __USE_BSD
#include <unistd.h>
#include <fcntl.h>
#define __USE_BSD
#include <errno.h>
#undef __USE_BSD

#include "global.h"
#include "genlib.h"
#include "reg.h"

#define MAXDEV	20

static int32 oldbits[MAXDEV], bitsleft[MAXDEV];

#ifdef DUMPREG
static FILE *dumpfile;		      /* REG dump file handle */
static unsigned char dumpbuf[1024];   /* REG dump buffer */
static int dumpptr = 0; 	      /* Pointer into dump buffer */
#endif

static int32 OpenDev(DevOpts *opts) {
  char ttydev[30];
  speed_t baudcon;
  struct termios tt;
  int32 TTY_fd, res;

#ifdef DUMPREG
  dumpfile = fopen("dumpreg.dat", "w");
  setbuf(dumpfile, NULL);
  dumpptr = 0;
#endif

#ifdef Solaris
  /* Serial ports (at least built-in ones) have names of
     /dev/term/a, /dev/term/b, etc.  Map the port numbers
     from the RC file into this nomenclature, with "1"
     designating /dev/term/a. */
#ifdef Irix
  sprintf(ttydev, "/dev/ttyd%d", opts->port);
#else
  sprintf(ttydev, "/dev/term/%c", 'a' + (opts->port - 1));
#endif
#else
  sprintf(ttydev, "/dev/ttyS%d", opts->port);
#endif

  switch(opts->baud) {
  case 1200: baudcon = B1200; break;
  case 2400: baudcon = B2400; break;
  case 4800: baudcon = B4800; break;
  case 9600: baudcon = B9600; break;
  case 19200: baudcon = B19200; break;
  case 38400: baudcon = B38400; break;
#ifndef Irix
  case 115200: baudcon = B115200; break;
#endif
  default:
    printf("%s: Baud rate %ld not supported.\n", pgmname, opts->baud);
    return -1;
  }

  fprintf(stderr, "Opening %s at %ld\n", ttydev, opts->baud);
  if ((TTY_fd = open(ttydev, O_RDONLY
#ifdef CPU_BOUND
				    | O_NDELAY
#endif
      )) < 0) {
    perror(pgmname);
    return -1;
  }

  if (TTY_fd >= MAXDEV) {
    fprintf(stderr, "%s: Too many devices open.\n", pgmname);
    close(TTY_fd);
    return -1;
  }

  res = tcgetattr(TTY_fd, &tt);
#ifdef Solaris
  tt.c_iflag = 0;
  tt.c_oflag = 0;
  tt.c_cflag = baudcon | CS8 | CREAD | CLOCAL;
  tt.c_lflag = 0;
  tt.c_cc[VMIN] = 1;	/* This many chars satisfies reads */
  tt.c_cc[VTIME] = 0;	/* or in this many tenths of seconds */
#else
  res = cfsetospeed(&tt, baudcon); 
  cfmakeraw(&tt);
  tt.c_oflag &= (~(TABDLY | ONLCR));
#endif
  res = tcsetattr(TTY_fd, TCSANOW, &tt);

  oldbits[TTY_fd] = bitsleft[TTY_fd] = 0;
  return TTY_fd;
}

static int32 Sample(int32 dd, uint16 bits) {
  int32 bc, sum, n1;
  uint8 c1;

  if (dd < 0) return -1;

  sum = bc = 0;
  while (bc < bits) {
    if (bitsleft[dd]) {
      sum += (oldbits[dd] & 0x01);
      oldbits[dd] >>= 1;
      bitsleft[dd]--;
      bc++;
    } else {
      do {
	n1 = read(dd, &c1, 1);
      } while (n1 == 0 || (n1 == -1 && errno == EAGAIN));
      if (n1 == -1) {
	/* Fatal error occurred, die now? */
	return -1;
      }
#ifdef DUMPREG
      dumpbuf[dumpptr++] = c1;
      if (dumpptr >= sizeof(dumpbuf)) {
	fwrite(dumpbuf, sizeof(dumpbuf), 1, dumpfile);
	dumpptr = 0;
      }
#endif
      oldbits[dd] = c1;
      bitsleft[dd] = 8;
    }
  }

  return sum;
}

#define SAMP_PERIOD	1000	/* msec */
#define MARGIN		.95	/* how much to headroom to allow in
				   speed measurement */

static int32 EvalSpeed(int32 dd) {
  struct timeval start, end;
  int32 bitct, samp;

  if (dd < 0) return -1;

  gettimeofday(&start, NULL);
  bitct = 0;
  while (1) {
    gettimeofday(&end, NULL);
    if (deltams(&end, &start) >= SAMP_PERIOD) break;
    samp = Sample(dd, 1);
    bitct++;
  }

  return (int32)(bitct * MARGIN);
}

static int32 Discard(int32 dd) {
  int32 disc;
#ifdef CPU_BOUND
  int32  n1;
  uint8 c1;
#endif
  
  if (dd < 0) return -1;

  disc = bitsleft[dd];
  bitsleft[dd] = 0;

#ifdef CPU_BOUND
  do {
    n1 = read(dd, &c1, 1);
    if (n1 == 1) disc++;
  } while (n1 == 1);
#else
  tcflush(dd, TCIFLUSH);
#endif

  return disc;
}

static int32 CloseDev(int32 dd) {
  if (dd < 0) return -1;

  close(dd);
  return 0;
}

/*  Driver description table.  */

REG_driver REG_pear = {
                        "PEAR", 
			0,
			OpenDev,
			EvalSpeed,
			Sample,
			Discard,
			CloseDev
		      };
