// This file is part of ICU4X. For terms of use, please see the file
// called LICENSE at the top level of the ICU4X source tree
// (online at: https://github.com/unicode-org/icu4x/blob/main/LICENSE ).

use litemap::LiteMap;

const fn single_byte_intermediate_value(x: u8) -> u8 {
    debug_assert!(x <= 0b00001111);
    x | 0b10000000
}

use single_byte_intermediate_value as single_byte_short_value;

const fn single_byte_branch_equal(x: u8) -> u8 {
    debug_assert!(x <= 0b00001111);
    x | 0b11000000
}

use single_byte_branch_equal as single_byte_short_match;

#[allow(dead_code)]
pub fn strings_to_litemap<'a>(strings: &[&'a str]) -> LiteMap<&'a [u8], usize> {
    strings
        .iter()
        .copied()
        .map(|x| x.as_bytes())
        .enumerate()
        .map(|(i, s)| (s, i))
        .collect()
}

#[allow(dead_code)]
pub mod basic {
    use super::*;
    pub static TRIE_ASCII: &[u8] = &[
        b'a',
        b'b',
        single_byte_short_value(1),
        b'c',
        single_byte_short_value(2),
        // Begin Match Node
        single_byte_short_match(3),
        b'd',
        b'e',
        b'f',
        5,
        8,
        // End Match Node
        // subslice @ 0
        single_byte_short_value(3),
        b'g',
        b'h',
        b'i',
        single_byte_short_value(4),
        // subslice @ 5
        b'j',
        b'k',
        single_byte_short_value(5),
        // subslice @ 8
        // Begin Match Node
        single_byte_short_match(2),
        b'l',
        b'm',
        1,
        // End Match Node
        // subsubslice @ 0
        single_byte_short_value(6),
        // subsubslice @ 1
        b'n',
        single_byte_short_value(7),
    ];
    pub static DATA_ASCII: &[(&[u8], usize)] = &[
        (b"ab", 1),
        (b"abc", 2),
        (b"abcd", 3),
        (b"abcdghi", 4),
        (b"abcejk", 5),
        (b"abcfl", 6),
        (b"abcfmn", 7),
    ];

    pub static TRIE_UNICODE: &[u8] = &[
        196, 100, 102, 103, 107, 12, 17, 23, 195, 97, 101, 105, 2, 4, 115, 129, 114, 130, 101, 131,
        162, 195, 188, 114, 132, 114, 111, 162, 195, 159, 133, 162, 195, 182, 110, 110, 101, 110,
        134,
    ];
    pub static DATA_UNICODE: &[(&[u8], usize)] = &[
        ("das".as_bytes(), 1),
        ("der".as_bytes(), 2),
        ("die".as_bytes(), 3),
        ("für".as_bytes(), 4),
        ("groß".as_bytes(), 5),
        ("können".as_bytes(), 6),
    ];

    pub static TRIE_BINARY: &[u8] = &[
        196, 0, 129, 144, 240, 3, 9, 12, 161, 144, 131, 194, 130, 131, 1, 129, 130, 161, 144, 132,
        161, 255, 133, 161, 255, 134,
    ];
    pub static DATA_BINARY: &[(&[u8], usize)] = &[
        (b"\0\x90", 3),
        (b"\x81\x82", 1),
        (b"\x81\x83", 2),
        (b"\x90\x90", 4),
        (b"\xF0\xFF", 5),
        (b"\xF0\xFF\xFF", 6),
    ];

    // Note: Cow and ZeroVec have the same serialized form
    pub static JSON_STR_ASCII: &str = "{\"trie\":{\"ab\":1,\"abc\":2,\"abcd\":3,\"abcdghi\":4,\"abcejk\":5,\"abcfl\":6,\"abcfmn\":7}}";
    pub static JSON_STR_UNICODE: &str =
        "{\"trie\":{\"das\":1,\"der\":2,\"die\":3,\"für\":4,\"groß\":5,\"können\":6}}";
    pub static JSON_STR_BINARY: &str = "{\"trie\":[[[0,144],3],[[129,130],1],[[129,131],2],[[144,144],4],[[240,255],5],[[240,255,255],6]]}";
    pub static BINCODE_BYTES_ASCII: &[u8] = &[
        97, 98, 129, 99, 130, 195, 100, 101, 102, 5, 8, 131, 103, 104,
        105, 132, 106, 107, 133, 194, 108, 109, 1, 134, 110, 135,
    ];
    pub static BINCODE_BYTES_UNICODE: &[u8] = &[
        196, 100, 102, 103, 107, 12, 17, 23, 195, 97, 101, 105, 2, 4, 115,
        129, 114, 130, 101, 131, 162, 195, 188, 114, 132, 114, 111, 162, 195, 159, 133, 162, 195,
        182, 110, 110, 101, 110, 134,
    ];
    pub static BINCODE_BYTES_BINARY: &[u8] = &[
        196, 0, 129, 144, 240, 3, 9, 12, 161, 144, 131, 194, 130, 131, 1,
        129, 130, 161, 144, 132, 161, 255, 133, 161, 255, 134,
    ];
}

#[allow(dead_code)]
pub mod short_subtags {
    pub static STRINGS: &[&str] = &[
        "aa",
        "aai",
        "aak",
        "aau",
        "ab",
        "abi",
        "abq",
        "abr",
        "abt",
        "aby",
        "acd",
        "ace",
        "ach",
        "ada",
        "ade",
        "adj",
        "adp",
        "ady",
        "adz",
        "ae",
        "aeb",
        "aey",
        "af",
        "agc",
        "agd",
        "agg",
        "agm",
        "ago",
        "agq",
        "aha",
        "ahl",
        "aho",
        "ajg",
        "ak",
        "akk",
        "ala",
        "ali",
        "aln",
        "alt",
        "am",
        "amm",
        "amn",
        "amo",
        "amp",
        "an",
        "anc",
        "ank",
        "ann",
        "any",
        "aoj",
        "aom",
        "aoz",
        "apc",
        "apd",
        "ape",
        "apr",
        "aps",
        "apz",
        "ar",
        "arc",
        "arc-Nbat",
        "arc-Palm",
        "arh",
        "arn",
        "aro",
        "arq",
        "ars",
        "ary",
        "arz",
        "as",
        "asa",
        "ase",
        "asg",
        "aso",
        "ast",
        "ata",
        "atg",
        "atj",
        "auy",
        "av",
        "avl",
        "avn",
        "avt",
        "avu",
        "awa",
        "awb",
        "awo",
        "awx",
        "ay",
        "ayb",
        "az",
        "az-Arab",
        "az-IQ",
        "az-IR",
        "az-RU",
        "ba",
        "bal",
        "ban",
        "bap",
        "bar",
        "bas",
        "bav",
        "bax",
        "bba",
        "bbb",
        "bbc",
        "bbd",
        "bbj",
        "bbp",
        "bbr",
        "bcf",
        "bch",
        "bci",
        "bcm",
        "bcn",
        "bco",
        "bcq",
        "bcu",
        "bdd",
        "be",
        "bef",
        "beh",
        "bej",
        "bem",
        "bet",
        "bew",
        "bex",
        "bez",
        "bfd",
        "bfq",
        "bft",
        "bfy",
        "bg",
        "bgc",
        "bgn",
        "bgx",
        "bhb",
        "bhg",
        "bhi",
        "bhl",
        "bho",
        "bhy",
        "bi",
        "bib",
        "big",
        "bik",
        "bim",
        "bin",
        "bio",
        "biq",
        "bjh",
        "bji",
        "bjj",
        "bjn",
        "bjo",
        "bjr",
        "bjt",
        "bjz",
        "bkc",
        "bkm",
        "bkq",
        "bku",
        "bkv",
        "bla",
        "blg",
        "blt",
        "bm",
        "bmh",
        "bmk",
        "bmq",
        "bmu",
        "bn",
        "bng",
        "bnm",
        "bnp",
        "bo",
        "boj",
        "bom",
        "bon",
        "bpy",
        "bqc",
        "bqi",
        "bqp",
        "bqv",
        "br",
        "bra",
        "brh",
        "brx",
        "brz",
        "bs",
        "bsj",
        "bsq",
        "bss",
        "bst",
        "bto",
        "btt",
        "btv",
        "bua",
        "buc",
        "bud",
        "bug",
        "buk",
        "bum",
        "buo",
        "bus",
        "buu",
        "bvb",
        "bwd",
        "bwr",
        "bxh",
        "bye",
        "byn",
        "byr",
        "bys",
        "byv",
        "byx",
        "bza",
        "bze",
        "bzf",
        "bzh",
        "bzw",
        "ca",
        "cad",
        "can",
        "cbj",
        "cch",
        "ccp",
        "ce",
        "ceb",
        "cfa",
        "cgg",
        "ch",
        "chk",
        "chm",
        "cho",
        "chp",
        "chr",
        "cic",
        "cja",
        "cjm",
        "cjv",
        "ckb",
        "ckl",
        "cko",
        "cky",
        "cla",
        "clc",
        "cme",
        "cmg",
        "co",
        "cop",
        "cps",
        "cr",
        "crg",
        "crh",
        "crk",
        "crl",
        "crs",
        "cs",
        "csb",
        "csw",
        "ctd",
        "cu",
        "cu-Glag",
        "cv",
        "cy",
        "da",
        "dad",
        "daf",
        "dag",
        "dah",
        "dak",
        "dar",
        "dav",
        "dbd",
        "dbq",
        "dcc",
        "ddn",
        "de",
        "ded",
        "den",
        "dga",
        "dgh",
        "dgi",
        "dgl",
        "dgr",
        "dgz",
        "dia",
        "dje",
        "dmf",
        "dnj",
        "dob",
        "doi",
        "dop",
        "dow",
        "drh",
        "dri",
        "drs",
        "dsb",
        "dtm",
        "dtp",
        "dts",
        "dty",
        "dua",
        "duc",
        "dud",
        "dug",
        "dv",
        "dva",
        "dww",
        "dyo",
        "dyu",
        "dz",
        "dzg",
        "ebu",
        "ee",
        "efi",
        "egl",
        "egy",
        "eka",
        "eky",
        "el",
        "ema",
        "emi",
        "en",
        "en-Shaw",
        "enn",
        "enq",
        "eo",
        "eri",
        "es",
        "esg",
        "esu",
        "et",
        "etr",
        "ett",
        "etu",
        "etx",
        "eu",
        "ewo",
        "ext",
        "eza",
        "fa",
        "faa",
        "fab",
        "fag",
        "fai",
        "fan",
        "ff",
        "ff-Adlm",
        "ffi",
        "ffm",
        "fi",
        "fia",
        "fil",
        "fit",
        "fj",
        "flr",
        "fmp",
        "fo",
        "fod",
        "fon",
        "for",
        "fpe",
        "fqs",
        "fr",
        "frc",
        "frp",
        "frr",
        "frs",
        "fub",
        "fud",
        "fue",
        "fuf",
        "fuh",
        "fuq",
        "fur",
        "fuv",
        "fuy",
        "fvr",
        "fy",
        "ga",
        "gaa",
        "gaf",
        "gag",
        "gah",
        "gaj",
        "gam",
        "gan",
        "gaw",
        "gay",
        "gba",
        "gbf",
        "gbm",
        "gby",
        "gbz",
        "gcr",
        "gd",
        "gde",
        "gdn",
        "gdr",
        "geb",
        "gej",
        "gel",
        "gez",
        "gfk",
        "ggn",
        "ghs",
        "gil",
        "gim",
        "gjk",
        "gjn",
        "gju",
        "gkn",
        "gkp",
        "gl",
        "glk",
        "gmm",
        "gmv",
        "gn",
        "gnd",
        "gng",
        "god",
        "gof",
        "goi",
        "gom",
        "gon",
        "gor",
        "gos",
        "got",
        "grb",
        "grc",
        "grc-Linb",
        "grt",
        "grw",
        "gsw",
        "gu",
        "gub",
        "guc",
        "gud",
        "gur",
        "guw",
        "gux",
        "guz",
        "gv",
        "gvf",
        "gvr",
        "gvs",
        "gwc",
        "gwi",
        "gwt",
        "gyi",
        "ha",
        "ha-CM",
        "ha-SD",
        "hag",
        "hak",
        "ham",
        "haw",
        "haz",
        "hbb",
        "hdy",
        "he",
        "hhy",
        "hi",
        "hi-Latn",
        "hia",
        "hif",
        "hig",
        "hih",
        "hil",
        "hla",
        "hlu",
        "hmd",
        "hmt",
        "hnd",
        "hne",
        "hnj",
        "hnn",
        "hno",
        "ho",
        "hoc",
        "hoj",
        "hot",
        "hr",
        "hsb",
        "hsn",
        "ht",
        "hu",
        "hui",
        "hur",
        "hy",
        "hz",
        "ia",
        "ian",
        "iar",
        "iba",
        "ibb",
        "iby",
        "ica",
        "ich",
        "id",
        "idd",
        "idi",
        "idu",
        "ife",
        "ig",
        "igb",
        "ige",
        "ii",
        "ijj",
        "ik",
        "ikk",
        "ikw",
        "ikx",
        "ilo",
        "imo",
        "in",
        "inh",
        "io",
        "iou",
        "iri",
        "is",
        "it",
        "iu",
        "iw",
        "iwm",
        "iws",
        "izh",
        "izi",
        "ja",
        "jab",
        "jam",
        "jar",
        "jbo",
        "jbu",
        "jen",
        "jgk",
        "jgo",
        "ji",
        "jib",
        "jmc",
        "jml",
        "jra",
        "jut",
        "jv",
        "jw",
        "ka",
        "kaa",
        "kab",
        "kac",
        "kad",
        "kai",
        "kaj",
        "kam",
        "kao",
        "kaw",
        "kbd",
        "kbm",
        "kbp",
        "kbq",
        "kbx",
        "kby",
        "kcg",
        "kck",
        "kcl",
        "kct",
        "kde",
        "kdh",
        "kdl",
        "kdt",
        "kea",
        "ken",
        "kez",
        "kfo",
        "kfr",
        "kfy",
        "kg",
        "kge",
        "kgf",
        "kgp",
        "kha",
        "khb",
        "khn",
        "khq",
        "khs",
        "kht",
        "khw",
        "khz",
        "ki",
        "kij",
        "kiu",
        "kiw",
        "kj",
        "kjd",
        "kjg",
        "kjs",
        "kjy",
        "kk",
        "kk-AF",
        "kk-Arab",
        "kk-CN",
        "kk-IR",
        "kk-MN",
        "kkc",
        "kkj",
        "kl",
        "kln",
        "klq",
        "klt",
        "klx",
        "km",
        "kmb",
        "kmh",
        "kmo",
        "kms",
        "kmu",
        "kmw",
        "kn",
        "knf",
        "knp",
        "ko",
        "koi",
        "kok",
        "kol",
        "kos",
        "koz",
        "kpe",
        "kpf",
        "kpo",
        "kpr",
        "kpx",
        "kqb",
        "kqf",
        "kqs",
        "kqy",
        "kr",
        "krc",
        "kri",
        "krj",
        "krl",
        "krs",
        "kru",
        "ks",
        "ksb",
        "ksd",
        "ksf",
        "ksh",
        "ksj",
        "ksr",
        "ktb",
        "ktm",
        "kto",
        "ktr",
        "ku",
        "ku-Arab",
        "ku-LB",
        "ku-Yezi",
        "kub",
        "kud",
        "kue",
        "kuj",
        "kum",
        "kun",
        "kup",
        "kus",
        "kv",
        "kvg",
        "kvr",
        "kvx",
        "kw",
        "kwj",
        "kwk",
        "kwo",
        "kwq",
        "kxa",
        "kxc",
        "kxe",
        "kxl",
        "kxm",
        "kxp",
        "kxw",
        "kxz",
        "ky",
        "ky-Arab",
        "ky-CN",
        "ky-Latn",
        "ky-TR",
        "kye",
        "kyx",
        "kzh",
        "kzj",
        "kzr",
        "kzt",
        "la",
        "lab",
        "lad",
        "lag",
        "lah",
        "laj",
        "las",
        "lb",
        "lbe",
        "lbu",
        "lbw",
        "lcm",
        "lcp",
        "ldb",
        "led",
        "lee",
        "lem",
        "lep",
        "leq",
        "leu",
        "lez",
        "lg",
        "lgg",
        "li",
        "lia",
        "lid",
        "lif",
        "lif-Limb",
        "lig",
        "lih",
        "lij",
        "lil",
        "lis",
        "ljp",
        "lki",
        "lkt",
        "lle",
        "lln",
        "lmn",
        "lmo",
        "lmp",
        "ln",
        "lns",
        "lnu",
        "lo",
        "loj",
        "lok",
        "lol",
        "lor",
        "los",
        "loz",
        "lrc",
        "lt",
        "ltg",
        "lu",
        "lua",
        "luo",
        "luy",
        "luz",
        "lv",
        "lwl",
        "lzh",
        "lzz",
        "mad",
        "maf",
        "mag",
        "mai",
        "mak",
        "man",
        "man-GN",
        "man-Nkoo",
        "mas",
        "maw",
        "maz",
        "mbh",
        "mbo",
        "mbq",
        "mbu",
        "mbw",
        "mci",
        "mcp",
        "mcq",
        "mcr",
        "mcu",
        "mda",
        "mde",
        "mdf",
        "mdh",
        "mdj",
        "mdr",
        "mdx",
        "med",
        "mee",
        "mek",
        "men",
        "mer",
        "met",
        "meu",
        "mfa",
        "mfe",
        "mfn",
        "mfo",
        "mfq",
        "mg",
        "mgh",
        "mgl",
        "mgo",
        "mgp",
        "mgy",
        "mh",
        "mhi",
        "mhl",
        "mi",
        "mic",
        "mif",
        "min",
        "miw",
        "mk",
        "mki",
        "mkl",
        "mkp",
        "mkw",
        "ml",
        "mle",
        "mlp",
        "mls",
        "mmo",
        "mmu",
        "mmx",
        "mn",
        "mn-CN",
        "mn-Mong",
        "mna",
        "mnf",
        "mni",
        "mnw",
        "mo",
        "moa",
        "moe",
        "moh",
        "mos",
        "mox",
        "mpp",
        "mps",
        "mpt",
        "mpx",
        "mql",
        "mr",
        "mrd",
        "mrj",
        "mro",
        "ms",
        "ms-CC",
        "mt",
        "mtc",
        "mtf",
        "mti",
        "mtr",
        "mua",
        "mur",
        "mus",
        "mva",
        "mvn",
        "mvy",
        "mwk",
        "mwr",
        "mwv",
        "mww",
        "mxc",
        "mxm",
        "my",
        "myk",
        "mym",
        "myv",
        "myw",
        "myx",
        "myz",
        "mzk",
        "mzm",
        "mzn",
        "mzp",
        "mzw",
        "mzz",
        "na",
        "nac",
        "naf",
        "nak",
        "nan",
        "nap",
        "naq",
        "nas",
        "nb",
        "nca",
        "nce",
        "ncf",
        "nch",
        "nco",
        "ncu",
        "nd",
        "ndc",
        "nds",
        "ne",
        "neb",
        "new",
        "nex",
        "nfr",
        "ng",
        "nga",
        "ngb",
        "ngl",
        "nhb",
        "nhe",
        "nhw",
        "nif",
        "nii",
        "nij",
        "nin",
        "niu",
        "niy",
        "niz",
        "njo",
        "nkg",
        "nko",
        "nl",
        "nmg",
        "nmz",
        "nn",
        "nnf",
        "nnh",
        "nnk",
        "nnm",
        "nnp",
        "no",
        "nod",
        "noe",
        "non",
        "nop",
        "nou",
        "nqo",
        "nr",
        "nrb",
        "nsk",
        "nsn",
        "nso",
        "nss",
        "nst",
        "ntm",
        "ntr",
        "nui",
        "nup",
        "nus",
        "nuv",
        "nux",
        "nv",
        "nwb",
        "nxq",
        "nxr",
        "ny",
        "nym",
        "nyn",
        "nzi",
        "oc",
        "ogc",
        "oj",
        "ojs",
        "oka",
        "okr",
        "okv",
        "om",
        "ong",
        "onn",
        "ons",
        "opm",
        "or",
        "oro",
        "oru",
        "os",
        "osa",
        "ota",
        "otk",
        "oui",
        "ozm",
        "pa",
        "pa-Arab",
        "pa-PK",
        "pag",
        "pal",
        "pal-Phlp",
        "pam",
        "pap",
        "pau",
        "pbi",
        "pcd",
        "pcm",
        "pdc",
        "pdt",
        "ped",
        "peo",
        "pex",
        "pfl",
        "phl",
        "phn",
        "pil",
        "pip",
        "pka",
        "pko",
        "pl",
        "pla",
        "pms",
        "png",
        "pnn",
        "pnt",
        "pon",
        "ppa",
        "ppo",
        "pqm",
        "pra",
        "prd",
        "prg",
        "ps",
        "pss",
        "pt",
        "ptp",
        "puu",
        "pwa",
        "qu",
        "quc",
        "qug",
        "rai",
        "raj",
        "rao",
        "rcf",
        "rej",
        "rel",
        "res",
        "rgn",
        "rhg",
        "ria",
        "rif",
        "rif-NL",
        "rjs",
        "rkt",
        "rm",
        "rmf",
        "rmo",
        "rmt",
        "rmu",
        "rn",
        "rna",
        "rng",
        "ro",
        "rob",
        "rof",
        "roo",
        "rro",
        "rtm",
        "ru",
        "rue",
        "rug",
        "rw",
        "rwk",
        "rwo",
        "ryu",
        "sa",
        "saf",
        "sah",
        "saq",
        "sas",
        "sat",
        "sav",
        "saz",
        "sba",
        "sbe",
        "sbp",
        "sc",
        "sck",
        "scl",
        "scn",
        "sco",
        "sd",
        "sd-Deva",
        "sd-IN",
        "sd-Khoj",
        "sd-Sind",
        "sdc",
        "sdh",
        "se",
        "sef",
        "seh",
        "sei",
        "ses",
        "sg",
        "sga",
        "sgs",
        "sgw",
        "sgz",
        "shi",
        "shk",
        "shn",
        "shu",
        "si",
        "sid",
        "sig",
        "sil",
        "sim",
        "sjr",
        "sk",
        "skc",
        "skr",
        "sks",
        "sl",
        "sld",
        "sli",
        "sll",
        "sly",
        "sm",
        "sma",
        "smj",
        "smn",
        "smp",
        "smq",
        "sms",
        "sn",
        "snc",
        "snk",
        "snp",
        "snx",
        "sny",
        "so",
        "sog",
        "sok",
        "soq",
        "sou",
        "soy",
        "spd",
        "spl",
        "sps",
        "sq",
        "sr",
        "sr-ME",
        "sr-RO",
        "sr-RU",
        "sr-TR",
        "srb",
        "srn",
        "srr",
        "srx",
        "ss",
        "ssd",
        "ssg",
        "ssy",
        "st",
        "stk",
        "stq",
        "su",
        "sua",
        "sue",
        "suk",
        "sur",
        "sus",
        "sv",
        "sw",
        "swb",
        "swc",
        "swg",
        "swp",
        "swv",
        "sxn",
        "sxw",
        "syl",
        "syr",
        "szl",
        "ta",
        "taj",
        "tal",
        "tan",
        "taq",
        "tbc",
        "tbd",
        "tbf",
        "tbg",
        "tbo",
        "tbw",
        "tbz",
        "tci",
        "tcy",
        "tdd",
        "tdg",
        "tdh",
        "tdu",
        "te",
        "ted",
        "tem",
        "teo",
        "tet",
        "tfi",
        "tg",
        "tg-Arab",
        "tg-PK",
        "tgc",
        "tgo",
        "tgu",
        "th",
        "thl",
        "thq",
        "thr",
        "ti",
        "tif",
        "tig",
        "tik",
        "tim",
        "tio",
        "tiv",
        "tk",
        "tkl",
        "tkr",
        "tkt",
        "tl",
        "tlf",
        "tlx",
        "tly",
        "tmh",
        "tmy",
        "tn",
        "tnh",
        "to",
        "tof",
        "tog",
        "toq",
        "tpi",
        "tpm",
        "tpz",
        "tqo",
        "tr",
        "tru",
        "trv",
        "trw",
        "ts",
        "tsd",
        "tsf",
        "tsg",
        "tsj",
        "tsw",
        "tt",
        "ttd",
        "tte",
        "ttj",
        "ttr",
        "tts",
        "ttt",
        "tuh",
        "tul",
        "tum",
        "tuq",
        "tvd",
        "tvl",
        "tvu",
        "twh",
        "twq",
        "txg",
        "txo",
        "ty",
        "tya",
        "tyv",
        "tzm",
        "ubu",
        "udi",
        "udm",
        "ug",
        "ug-Cyrl",
        "ug-KZ",
        "ug-MN",
        "uga",
        "uk",
        "uli",
        "umb",
        "und",
        "und-002",
        "und-003",
        "und-005",
        "und-009",
        "und-011",
        "und-013",
        "und-014",
        "und-015",
        "und-017",
        "und-018",
        "und-019",
        "und-021",
        "und-029",
        "und-030",
        "und-034",
        "und-035",
        "und-039",
        "und-053",
        "und-054",
        "und-057",
        "und-061",
        "und-142",
        "und-143",
        "und-145",
        "und-150",
        "und-151",
        "und-154",
        "und-155",
        "und-202",
        "und-419",
        "und-AD",
        "und-Adlm",
        "und-AE",
        "und-AF",
        "und-Aghb",
        "und-Ahom",
        "und-AL",
        "und-AM",
        "und-AO",
        "und-AQ",
        "und-AR",
        "und-Arab",
        "und-Arab-CC",
        "und-Arab-CN",
        "und-Arab-GB",
        "und-Arab-ID",
        "und-Arab-IN",
        "und-Arab-KH",
        "und-Arab-MM",
        "und-Arab-MN",
        "und-Arab-MU",
        "und-Arab-NG",
        "und-Arab-PK",
        "und-Arab-TG",
        "und-Arab-TH",
        "und-Arab-TJ",
        "und-Arab-TR",
        "und-Arab-YT",
        "und-Armi",
        "und-Armn",
        "und-AS",
        "und-AT",
        "und-Avst",
        "und-AW",
        "und-AX",
        "und-AZ",
        "und-BA",
        "und-Bali",
        "und-Bamu",
        "und-Bass",
        "und-Batk",
        "und-BD",
        "und-BE",
        "und-Beng",
        "und-BF",
        "und-BG",
        "und-BH",
        "und-Bhks",
        "und-BI",
        "und-BJ",
        "und-BL",
        "und-BN",
        "und-BO",
        "und-Bopo",
        "und-BQ",
        "und-BR",
        "und-Brah",
        "und-Brai",
        "und-BT",
        "und-Bugi",
        "und-Buhd",
        "und-BV",
        "und-BY",
        "und-Cakm",
        "und-Cans",
        "und-Cari",
        "und-CD",
        "und-CF",
        "und-CG",
        "und-CH",
        "und-Cham",
        "und-Cher",
        "und-Chrs",
        "und-CI",
        "und-CL",
        "und-CM",
        "und-CN",
        "und-CO",
        "und-Copt",
        "und-CP",
        "und-Cpmn",
        "und-Cpmn-CY",
        "und-Cprt",
        "und-CR",
        "und-CU",
        "und-CV",
        "und-CW",
        "und-CY",
        "und-Cyrl",
        "und-Cyrl-AL",
        "und-Cyrl-BA",
        "und-Cyrl-GE",
        "und-Cyrl-GR",
        "und-Cyrl-MD",
        "und-Cyrl-RO",
        "und-Cyrl-SK",
        "und-Cyrl-TR",
        "und-Cyrl-XK",
        "und-CZ",
        "und-DE",
        "und-Deva",
        "und-Deva-BT",
        "und-Deva-FJ",
        "und-Deva-MU",
        "und-Deva-PK",
        "und-Diak",
        "und-DJ",
        "und-DK",
        "und-DO",
        "und-Dogr",
        "und-Dupl",
        "und-DZ",
        "und-EA",
        "und-EC",
        "und-EE",
        "und-EG",
        "und-Egyp",
        "und-EH",
        "und-Elba",
        "und-Elym",
        "und-ER",
        "und-ES",
        "und-ET",
        "und-Ethi",
        "und-EU",
        "und-EZ",
        "und-FI",
        "und-FO",
        "und-FR",
        "und-GA",
        "und-GE",
        "und-Geor",
        "und-GF",
        "und-GH",
        "und-GL",
        "und-Glag",
        "und-GN",
        "und-Gong",
        "und-Gonm",
        "und-Goth",
        "und-GP",
        "und-GQ",
        "und-GR",
        "und-Gran",
        "und-Grek",
        "und-Grek-TR",
        "und-GS",
        "und-GT",
        "und-Gujr",
        "und-Guru",
        "und-GW",
        "und-Hanb",
        "und-Hang",
        "und-Hani",
        "und-Hano",
        "und-Hans",
        "und-Hant",
        "und-Hant-CA",
        "und-Hebr",
        "und-Hebr-SE",
        "und-Hebr-UA",
        "und-Hebr-US",
        "und-Hira",
        "und-HK",
        "und-Hluw",
        "und-HM",
        "und-Hmng",
        "und-Hmnp",
        "und-HN",
        "und-HR",
        "und-HT",
        "und-HU",
        "und-Hung",
        "und-IC",
        "und-ID",
        "und-IL",
        "und-IN",
        "und-IQ",
        "und-IR",
        "und-IS",
        "und-IT",
        "und-Ital",
        "und-Jamo",
        "und-Java",
        "und-JO",
        "und-JP",
        "und-Jpan",
        "und-Kali",
        "und-Kana",
        "und-Kawi",
        "und-KE",
        "und-KG",
        "und-KH",
        "und-Khar",
        "und-Khmr",
        "und-Khoj",
        "und-Kits",
        "und-KM",
        "und-Knda",
        "und-Kore",
        "und-KP",
        "und-KR",
        "und-Kthi",
        "und-KW",
        "und-KZ",
        "und-LA",
        "und-Lana",
        "und-Laoo",
        "und-Latn-AF",
        "und-Latn-AM",
        "und-Latn-CN",
        "und-Latn-CY",
        "und-Latn-DZ",
        "und-Latn-ET",
        "und-Latn-GE",
        "und-Latn-IR",
        "und-Latn-KM",
        "und-Latn-MA",
        "und-Latn-MK",
        "und-Latn-MM",
        "und-Latn-MO",
        "und-Latn-MR",
        "und-Latn-RU",
        "und-Latn-SY",
        "und-Latn-TN",
        "und-Latn-TW",
        "und-Latn-UA",
        "und-LB",
        "und-Lepc",
        "und-LI",
        "und-Limb",
        "und-Lina",
        "und-Linb",
        "und-Lisu",
        "und-LK",
        "und-LS",
        "und-LT",
        "und-LU",
        "und-LV",
        "und-LY",
        "und-Lyci",
        "und-Lydi",
        "und-MA",
        "und-Mahj",
        "und-Maka",
        "und-Mand",
        "und-Mani",
        "und-Marc",
        "und-MC",
        "und-MD",
        "und-ME",
        "und-Medf",
        "und-Mend",
        "und-Merc",
        "und-Mero",
        "und-MF",
        "und-MG",
        "und-MK",
        "und-ML",
        "und-Mlym",
        "und-MM",
        "und-MN",
        "und-MO",
        "und-Modi",
        "und-Mong",
        "und-MQ",
        "und-MR",
        "und-Mroo",
        "und-MT",
        "und-Mtei",
        "und-MU",
        "und-Mult",
        "und-MV",
        "und-MX",
        "und-MY",
        "und-Mymr",
        "und-Mymr-IN",
        "und-Mymr-TH",
        "und-MZ",
        "und-NA",
        "und-Nagm",
        "und-Nand",
        "und-Narb",
        "und-Nbat",
        "und-NC",
        "und-NE",
        "und-Newa",
        "und-NI",
        "und-Nkoo",
        "und-NL",
        "und-NO",
        "und-NP",
        "und-Nshu",
        "und-Ogam",
        "und-Olck",
        "und-OM",
        "und-Orkh",
        "und-Orya",
        "und-Osge",
        "und-Osma",
        "und-Ougr",
        "und-PA",
        "und-Palm",
        "und-Pauc",
        "und-PE",
        "und-Perm",
        "und-PF",
        "und-PG",
        "und-PH",
        "und-Phag",
        "und-Phli",
        "und-Phlp",
        "und-Phnx",
        "und-PK",
        "und-PL",
        "und-Plrd",
        "und-PM",
        "und-PR",
        "und-Prti",
        "und-PS",
        "und-PT",
        "und-PW",
        "und-PY",
        "und-QA",
        "und-QO",
        "und-RE",
        "und-Rjng",
        "und-RO",
        "und-Rohg",
        "und-RS",
        "und-RU",
        "und-Runr",
        "und-RW",
        "und-SA",
        "und-Samr",
        "und-Sarb",
        "und-Saur",
        "und-SC",
        "und-SD",
        "und-SE",
        "und-Sgnw",
        "und-Shaw",
        "und-Shrd",
        "und-SI",
        "und-Sidd",
        "und-Sind",
        "und-Sinh",
        "und-SJ",
        "und-SK",
        "und-SM",
        "und-SN",
        "und-SO",
        "und-Sogd",
        "und-Sogo",
        "und-Sora",
        "und-Soyo",
        "und-SR",
        "und-ST",
        "und-Sund",
        "und-SV",
        "und-SY",
        "und-Sylo",
        "und-Syrc",
        "und-Tagb",
        "und-Takr",
        "und-Tale",
        "und-Talu",
        "und-Taml",
        "und-Tang",
        "und-Tavt",
        "und-TD",
        "und-Telu",
        "und-TF",
        "und-Tfng",
        "und-TG",
        "und-Tglg",
        "und-TH",
        "und-Thaa",
        "und-Thai",
        "und-Thai-CN",
        "und-Thai-KH",
        "und-Thai-LA",
        "und-Tibt",
        "und-Tirh",
        "und-TJ",
        "und-TK",
        "und-TL",
        "und-TM",
        "und-TN",
        "und-Tnsa",
        "und-TO",
        "und-Toto",
        "und-TR",
        "und-TV",
        "und-TW",
        "und-TZ",
        "und-UA",
        "und-UG",
        "und-Ugar",
        "und-UY",
        "und-UZ",
        "und-VA",
        "und-Vaii",
        "und-VE",
        "und-Vith",
        "und-VN",
        "und-VU",
        "und-Wara",
        "und-Wcho",
        "und-WF",
        "und-WS",
        "und-XK",
        "und-Xpeo",
        "und-Xsux",
        "und-YE",
        "und-Yezi",
        "und-Yiii",
        "und-YT",
        "und-Zanb",
        "und-ZW",
        "unr",
        "unr-Deva",
        "unr-NP",
        "unx",
        "uok",
        "ur",
        "uri",
        "urt",
        "urw",
        "usa",
        "uth",
        "utr",
        "uvh",
        "uvl",
        "uz",
        "uz-AF",
        "uz-Arab",
        "uz-CN",
        "vag",
        "vai",
        "van",
        "ve",
        "vec",
        "vep",
        "vi",
        "vic",
        "viv",
        "vls",
        "vmf",
        "vmw",
        "vo",
        "vot",
        "vro",
        "vun",
        "vut",
        "wa",
        "wae",
        "waj",
        "wal",
        "wan",
        "war",
        "wbp",
        "wbq",
        "wbr",
        "wci",
        "wer",
        "wgi",
        "whg",
        "wib",
        "wiu",
        "wiv",
        "wja",
        "wji",
        "wls",
        "wmo",
        "wnc",
        "wni",
        "wnu",
        "wo",
        "wob",
        "wos",
        "wrs",
        "wsg",
        "wsk",
        "wtm",
        "wuu",
        "wuv",
        "wwa",
        "xav",
        "xbi",
        "xco",
        "xcr",
        "xes",
        "xh",
        "xla",
        "xlc",
        "xld",
        "xmf",
        "xmn",
        "xmr",
        "xna",
        "xnr",
        "xog",
        "xon",
        "xpr",
        "xrb",
        "xsa",
        "xsi",
        "xsm",
        "xsr",
        "xwe",
        "yam",
        "yao",
        "yap",
        "yas",
        "yat",
        "yav",
        "yay",
        "yaz",
        "yba",
        "ybb",
        "yby",
        "yer",
        "ygr",
        "ygw",
        "yi",
        "yko",
        "yle",
        "ylg",
        "yll",
        "yml",
        "yo",
        "yon",
        "yrb",
        "yre",
        "yrl",
        "yss",
        "yua",
        "yue",
        "yue-CN",
        "yue-Hans",
        "yuj",
        "yut",
        "yuw",
        "za",
        "zag",
        "zdj",
        "zea",
        "zgh",
        "zh",
        "zh-AU",
        "zh-BN",
        "zh-Bopo",
        "zh-GB",
        "zh-GF",
        "zh-Hanb",
        "zh-Hant",
        "zh-HK",
        "zh-ID",
        "zh-MO",
        "zh-PA",
        "zh-PF",
        "zh-PH",
        "zh-SR",
        "zh-TH",
        "zh-TW",
        "zh-US",
        "zh-VN",
        "zhx",
        "zia",
        "zkt",
        "zlm",
        "zmi",
        "zne",
        "zu",
        "zza",
    ];
}

#[allow(dead_code)]
pub mod short_subtags_10pct {
    pub static STRINGS: &[&str] = &[
        "aa",
        "acd",
        "aeb",
        "ahl",
        "amm",
        "aom",
        "arc-Nbat",
        "asa",
        "avl",
        "az",
        "bas",
        "bcf",
        "bef",
        "bft",
        "bho",
        "bjh",
        "bkq",
        "bmu",
        "bqc",
        "bsj",
        "bug",
        "bye",
        "bzw",
        "cgg",
        "cjv",
        "cop",
        "csw",
        "dah",
        "den",
        "dnj",
        "dtp",
        "dyo",
        "eky",
        "es",
        "ext",
        "ffi",
        "fod",
        "fub",
        "fy",
        "gay",
        "gdr",
        "gjk",
        "gnd",
        "grb",
        "gur",
        "gwt",
        "hbb",
        "hil",
        "ho",
        "hur",
        "ich",
        "ijj",
        "iou",
        "ja",
        "jib",
        "kac",
        "kbq",
        "kdt",
        "kgp",
        "kij",
        "kk-Arab",
        "klx",
        "knp",
        "kpr",
        "krl",
        "ktb",
        "kue",
        "kw",
        "kxp",
        "kzh",
        "las",
        "lem",
        "lif",
        "lle",
        "lok",
        "luo",
        "mai",
        "mbq",
        "mdf",
        "met",
        "mgo",
        "miw",
        "mmo",
        "mo",
        "mql",
        "mti",
        "mwv",
        "myz",
        "nak",
        "nco",
        "ng",
        "nin",
        "nn",
        "nop",
        "ntm",
        "nxr",
        "okr",
        "os",
        "pal",
        "ped",
        "pl",
        "pra",
        "quc",
        "rhg",
        "rmu",
        "ru",
        "saq",
        "scl",
        "se",
        "shi",
        "sk",
        "sma",
        "snx",
        "sps",
        "srx",
        "sue",
        "swv",
        "taq",
        "tdd",
        "tg",
        "ti",
        "tkt",
        "tof",
        "trw",
        "ttj",
        "tvu",
        "udi",
        "und",
        "und-018",
        "und-057",
        "und-419",
        "und-AQ",
        "und-Arab-MN",
        "und-Armn",
        "und-Bass",
        "und-BJ",
        "und-Bugi",
        "und-CH",
        "und-CP",
        "und-Cyrl-AL",
        "und-DE",
        "und-Dogr",
        "und-Elym",
        "und-GA",
        "und-Goth",
        "und-Guru",
        "und-Hebr-SE",
        "und-HR",
        "und-IS",
        "und-Kawi",
        "und-Kore",
        "und-Latn-AM",
        "und-Latn-MM",
        "und-LI",
        "und-LY",
        "und-MD",
        "und-Mlym",
        "und-Mtei",
        "und-NA",
        "und-NL",
        "und-Osma",
        "und-Phag",
        "und-PS",
        "und-RS",
        "und-SE",
        "und-SM",
        "und-SV",
        "und-Tavt",
        "und-Thai-CN",
        "und-Tnsa",
        "und-UY",
        "und-WF",
        "und-ZW",
        "usa",
        "vai",
        "vmw",
        "wan",
        "wiu",
        "wob",
        "xbi",
        "xmr",
        "xsr",
        "yba",
        "yll",
        "yue-CN",
        "zh",
        "zh-MO",
        "zia",
    ];
}

#[allow(dead_code)]
pub mod locales_with_aux {
    pub static NUM_UNIQUE_BLOBS: usize = 411;
    pub static STRINGS: &[&str] = &["af-x-3", "af-x-3s", "af-x-4", "af-x-4s", "af-x-5", "af-x-5s", "am-x-3", "am-x-3s", "am-x-4", "am-x-4s", "am-x-5", "am-x-5s", "ar-DZ-x-3", "ar-DZ-x-3s", "ar-DZ-x-4", "ar-DZ-x-4s", "ar-DZ-x-5", "ar-DZ-x-5s", "ar-IQ-x-3", "ar-IQ-x-3s", "ar-IQ-x-4", "ar-IQ-x-4s", "ar-IQ-x-5", "ar-IQ-x-5s", "ar-JO-x-3", "ar-JO-x-3s", "ar-JO-x-4", "ar-JO-x-4s", "ar-JO-x-5", "ar-JO-x-5s", "ar-LB-x-3", "ar-LB-x-3s", "ar-LB-x-4", "ar-LB-x-4s", "ar-LB-x-5", "ar-LB-x-5s", "ar-MA-x-3", "ar-MA-x-3s", "ar-MA-x-4", "ar-MA-x-4s", "ar-MA-x-5", "ar-MA-x-5s", "ar-MR-x-3", "ar-MR-x-3s", "ar-MR-x-4", "ar-MR-x-4s", "ar-MR-x-5", "ar-MR-x-5s", "ar-PS-x-3", "ar-PS-x-3s", "ar-PS-x-4", "ar-PS-x-4s", "ar-PS-x-5", "ar-PS-x-5s", "ar-SY-x-3", "ar-SY-x-3s", "ar-SY-x-4", "ar-SY-x-4s", "ar-SY-x-5", "ar-SY-x-5s", "ar-TN-x-3", "ar-TN-x-3s", "ar-TN-x-4", "ar-TN-x-4s", "ar-TN-x-5", "ar-TN-x-5s", "ar-x-3", "ar-x-3s", "ar-x-4", "ar-x-4s", "ar-x-5", "ar-x-5s", "as-x-3", "as-x-3s", "as-x-4", "as-x-4s", "as-x-5", "as-x-5s", "ast-x-3", "ast-x-3s", "ast-x-4", "ast-x-4s", "ast-x-5", "ast-x-5s", "az-x-3", "az-x-3s", "az-x-5", "az-x-5s", "be-x-3", "be-x-3s", "be-x-4", "be-x-4s", "be-x-5", "be-x-5s", "bg-x-3", "bg-x-3s", "bg-x-4", "bg-x-4s", "bg-x-5", "bg-x-5s", "bgc-x-3", "bgc-x-3s", "bgc-x-5", "bgc-x-5s", "bho-x-3", "bho-x-3s", "bho-x-5", "bho-x-5s", "bn-IN-x-3", "bn-IN-x-3s", "bn-IN-x-4", "bn-IN-x-4s", "bn-x-3", "bn-x-3s", "bn-x-4", "bn-x-4s", "bn-x-5", "bn-x-5s", "br-x-3", "br-x-3s", "br-x-4", "br-x-4s", "br-x-5", "br-x-5s", "brx-x-3", "brx-x-3s", "brx-x-4", "brx-x-4s", "brx-x-5", "brx-x-5s", "bs-Cyrl-x-3", "bs-Cyrl-x-3s", "bs-Cyrl-x-4", "bs-Cyrl-x-4s", "bs-Cyrl-x-5", "bs-Cyrl-x-5s", "bs-x-3", "bs-x-3s", "bs-x-4", "bs-x-4s", "bs-x-5", "bs-x-5s", "ca-x-3", "ca-x-3s", "ca-x-4", "ca-x-4s", "ca-x-5", "ca-x-5s", "ceb-x-3", "ceb-x-3s", "ceb-x-4", "ceb-x-4s", "ceb-x-5", "ceb-x-5s", "chr-x-3", "chr-x-3s", "chr-x-4", "chr-x-4s", "chr-x-5", "chr-x-5s", "cs-x-3", "cs-x-3s", "cs-x-5", "cs-x-5s", "cv-x-3", "cv-x-3s", "cv-x-4", "cv-x-4s", "cv-x-5", "cv-x-5s", "cy-x-3", "cy-x-3s", "cy-x-4", "cy-x-4s", "cy-x-5", "cy-x-5s", "da-x-3", "da-x-3s", "da-x-4", "da-x-4s", "da-x-5", "da-x-5s", "de-AT-x-3", "de-AT-x-3s", "de-AT-x-5", "de-AT-x-5s", "de-IT-x-3", "de-IT-x-3s", "de-IT-x-5", "de-IT-x-5s", "de-x-3", "de-x-3s", "de-x-4", "de-x-4s", "de-x-5", "de-x-5s", "doi-x-3", "doi-x-3s", "doi-x-4", "doi-x-4s", "doi-x-5", "doi-x-5s", "dsb-x-3", "dsb-x-3s", "dsb-x-4", "dsb-x-4s", "dsb-x-5", "dsb-x-5s", "el-polyton-x-3", "el-polyton-x-5", "el-polyton-x-5s", "el-x-3", "el-x-3s", "el-x-4", "el-x-4s", "el-x-5", "el-x-5s", "en-001-x-3", "en-001-x-3s", "en-150-x-3", "en-150-x-3s", "en-AG-x-3", "en-AG-x-3s", "en-AI-x-3", "en-AI-x-3s", "en-AT-x-3", "en-AT-x-3s", "en-AU-x-3", "en-AU-x-3s", "en-BB-x-3", "en-BB-x-3s", "en-BE-x-3", "en-BE-x-3s", "en-BM-x-3", "en-BM-x-3s", "en-BS-x-3", "en-BS-x-3s", "en-BW-x-3", "en-BW-x-3s", "en-BZ-x-3", "en-BZ-x-3s", "en-CC-x-3", "en-CC-x-3s", "en-CH-x-3", "en-CH-x-3s", "en-CK-x-3", "en-CK-x-3s", "en-CM-x-3", "en-CM-x-3s", "en-CX-x-3", "en-CX-x-3s", "en-CY-x-3", "en-CY-x-3s", "en-DE-x-3", "en-DE-x-3s", "en-DG-x-3", "en-DG-x-3s", "en-DK-x-3", "en-DK-x-3s", "en-DM-x-3", "en-DM-x-3s", "en-ER-x-3", "en-ER-x-3s", "en-FI-x-3", "en-FI-x-3s", "en-FJ-x-3", "en-FJ-x-3s", "en-FK-x-3", "en-FK-x-3s", "en-FM-x-3", "en-FM-x-3s", "en-GB-x-3", "en-GB-x-3s", "en-GD-x-3", "en-GD-x-3s", "en-GG-x-3", "en-GG-x-3s", "en-GH-x-3", "en-GH-x-3s", "en-GI-x-3", "en-GI-x-3s", "en-GM-x-3", "en-GM-x-3s", "en-GY-x-3", "en-GY-x-3s", "en-HK-x-3", "en-HK-x-3s", "en-IE-x-3", "en-IE-x-3s", "en-IL-x-3", "en-IL-x-3s", "en-IM-x-3", "en-IM-x-3s", "en-IN-x-3", "en-IN-x-3s", "en-IO-x-3", "en-IO-x-3s", "en-JE-x-3", "en-JE-x-3s", "en-JM-x-3", "en-JM-x-3s", "en-KE-x-3", "en-KE-x-3s", "en-KI-x-3", "en-KI-x-3s", "en-KN-x-3", "en-KN-x-3s", "en-KY-x-3", "en-KY-x-3s", "en-LC-x-3", "en-LC-x-3s", "en-LR-x-3", "en-LR-x-3s", "en-LS-x-3", "en-LS-x-3s", "en-MG-x-3", "en-MG-x-3s", "en-MO-x-3", "en-MO-x-3s", "en-MS-x-3", "en-MS-x-3s", "en-MT-x-3", "en-MT-x-3s", "en-MU-x-3", "en-MU-x-3s", "en-MV-x-3", "en-MV-x-3s", "en-MW-x-3", "en-MW-x-3s", "en-MY-x-3", "en-MY-x-3s", "en-NA-x-3", "en-NA-x-3s", "en-NF-x-3", "en-NF-x-3s", "en-NG-x-3", "en-NG-x-3s", "en-NL-x-3", "en-NL-x-3s", "en-NR-x-3", "en-NR-x-3s", "en-NU-x-3", "en-NU-x-3s", "en-NZ-x-3", "en-NZ-x-3s", "en-PG-x-3", "en-PG-x-3s", "en-PK-x-3", "en-PK-x-3s", "en-PN-x-3", "en-PN-x-3s", "en-PW-x-3", "en-PW-x-3s", "en-RW-x-3", "en-RW-x-3s", "en-SB-x-3", "en-SB-x-3s", "en-SC-x-3", "en-SC-x-3s", "en-SD-x-3", "en-SD-x-3s", "en-SE-x-3", "en-SE-x-3s", "en-SG-x-3", "en-SG-x-3s", "en-SH-x-3", "en-SH-x-3s", "en-SI-x-3", "en-SI-x-3s", "en-SL-x-3", "en-SL-x-3s", "en-SS-x-3", "en-SS-x-3s", "en-SX-x-3", "en-SX-x-3s", "en-SZ-x-3", "en-SZ-x-3s", "en-TC-x-3", "en-TC-x-3s", "en-TK-x-3", "en-TK-x-3s", "en-TO-x-3", "en-TO-x-3s", "en-TT-x-3", "en-TT-x-3s", "en-TV-x-3", "en-TV-x-3s", "en-TZ-x-3", "en-TZ-x-3s", "en-UG-x-3", "en-UG-x-3s", "en-VC-x-3", "en-VC-x-3s", "en-VG-x-3", "en-VG-x-3s", "en-VU-x-3", "en-VU-x-3s", "en-WS-x-3", "en-WS-x-3s", "en-ZA-x-3", "en-ZA-x-3s", "en-ZM-x-3", "en-ZM-x-3s", "en-ZW-x-3", "en-ZW-x-3s", "en-x-3", "en-x-3s", "en-x-4", "en-x-4s", "en-x-5", "en-x-5s", "es-CL-x-3s", "es-CO-x-3s", "es-PE-x-3", "es-PE-x-3s", "es-PE-x-5", "es-PE-x-5s", "es-PY-x-3", "es-PY-x-3s", "es-UY-x-3", "es-UY-x-3s", "es-UY-x-5", "es-UY-x-5s", "es-VE-x-3", "es-VE-x-3s", "es-x-3", "es-x-3s", "es-x-4", "es-x-4s", "es-x-5", "es-x-5s", "et-x-3", "et-x-3s", "et-x-4", "et-x-4s", "et-x-5", "et-x-5s", "eu-x-3", "eu-x-3s", "eu-x-4", "eu-x-4s", "eu-x-5", "eu-x-5s", "fa-AF-x-3", "fa-AF-x-3s", "fa-AF-x-4", "fa-AF-x-4s", "fa-AF-x-5", "fa-AF-x-5s", "fa-x-3", "fa-x-3s", "fa-x-4", "fa-x-4s", "fa-x-5", "fa-x-5s", "ff-Adlm-x-3", "ff-Adlm-x-3s", "ff-Adlm-x-4", "ff-Adlm-x-4s", "ff-Adlm-x-5", "ff-Adlm-x-5s", "fi-x-3", "fi-x-3s", "fi-x-4", "fi-x-4s", "fi-x-5", "fi-x-5s", "fil-x-3", "fil-x-3s", "fil-x-4", "fil-x-4s", "fil-x-5", "fil-x-5s", "fo-x-3", "fo-x-3s", "fo-x-4", "fo-x-4s", "fo-x-5", "fo-x-5s", "fr-CA-x-3", "fr-CA-x-3s", "fr-MA-x-3", "fr-MA-x-3s", "fr-x-3", "fr-x-3s", "fr-x-4", "fr-x-4s", "fr-x-5", "fr-x-5s", "ga-x-3", "ga-x-3s", "ga-x-4", "ga-x-4s", "ga-x-5", "ga-x-5s", "gd-x-3", "gd-x-3s", "gd-x-4", "gd-x-4s", "gd-x-5", "gd-x-5s", "gl-x-3", "gl-x-3s", "gl-x-4", "gl-x-4s", "gl-x-5", "gl-x-5s", "gu-x-3", "gu-x-3s", "gu-x-4", "gu-x-4s", "gu-x-5", "gu-x-5s", "ha-x-3", "ha-x-3s", "ha-x-4", "ha-x-4s", "ha-x-5", "ha-x-5s", "he-x-3", "he-x-3s", "he-x-5", "he-x-5s", "hi-Latn-x-3", "hi-Latn-x-3s", "hi-Latn-x-4", "hi-Latn-x-4s", "hi-Latn-x-5", "hi-Latn-x-5s", "hi-x-3", "hi-x-3s", "hi-x-4", "hi-x-4s", "hi-x-5", "hi-x-5s", "hr-x-3", "hr-x-3s", "hr-x-4", "hr-x-4s", "hr-x-5", "hr-x-5s", "hsb-x-3", "hsb-x-3s", "hsb-x-4", "hsb-x-4s", "hsb-x-5", "hsb-x-5s", "hu-x-3", "hu-x-3s", "hu-x-4", "hu-x-4s", "hu-x-5", "hu-x-5s", "hy-x-3", "hy-x-3s", "hy-x-4", "hy-x-4s", "hy-x-5", "hy-x-5s", "ia-x-3", "ia-x-3s", "ia-x-4", "ia-x-5", "ia-x-5s", "id-x-3", "id-x-3s", "id-x-4", "id-x-4s", "id-x-5", "id-x-5s", "ig-x-3", "ig-x-3s", "ig-x-4", "ig-x-4s", "ig-x-5", "ig-x-5s", "is-x-3", "is-x-3s", "is-x-4", "is-x-4s", "is-x-5", "is-x-5s", "it-x-3", "it-x-3s", "it-x-4", "it-x-4s", "it-x-5", "it-x-5s", "ja-x-3", "ja-x-3s", "ja-x-5", "ja-x-5s", "jv-x-3", "jv-x-3s", "jv-x-4", "jv-x-4s", "jv-x-5", "jv-x-5s", "ka-x-3", "ka-x-3s", "ka-x-4", "ka-x-4s", "ka-x-5", "ka-x-5s", "kea-x-3", "kea-x-3s", "kea-x-4", "kea-x-4s", "kea-x-5", "kea-x-5s", "kgp-x-3", "kgp-x-3s", "kgp-x-4", "kgp-x-4s", "kgp-x-5", "kgp-x-5s", "kk-x-3", "kk-x-3s", "kk-x-4", "kk-x-4s", "kk-x-5", "kk-x-5s", "km-x-3", "km-x-3s", "km-x-4", "km-x-4s", "km-x-5", "km-x-5s", "kn-x-3", "kn-x-3s", "kn-x-4", "kn-x-4s", "kn-x-5", "kn-x-5s", "ko-x-3", "ko-x-3s", "ko-x-4", "ko-x-4s", "ko-x-5", "ko-x-5s", "kok-x-3", "kok-x-3s", "kok-x-5", "kok-x-5s", "ks-Deva-x-3", "ks-Deva-x-3s", "ks-Deva-x-4", "ks-Deva-x-4s", "ks-Deva-x-5", "ks-Deva-x-5s", "ks-x-3", "ks-x-3s", "ks-x-4", "ks-x-4s", "ks-x-5", "ks-x-5s", "ky-x-3", "ky-x-3s", "ky-x-4", "ky-x-4s", "ky-x-5", "ky-x-5s", "lo-x-3", "lo-x-3s", "lo-x-5", "lo-x-5s", "lt-x-3", "lt-x-3s", "lt-x-4", "lt-x-4s", "lt-x-5", "lt-x-5s", "lv-x-3", "lv-x-3s", "lv-x-4", "lv-x-4s", "lv-x-5", "lv-x-5s", "mai-x-3", "mai-x-3s", "mai-x-4", "mai-x-4s", "mai-x-5", "mai-x-5s", "mi-x-3", "mi-x-3s", "mi-x-4", "mi-x-4s", "mi-x-5", "mi-x-5s", "mk-x-3", "mk-x-3s", "mk-x-4", "mk-x-4s", "mk-x-5", "mk-x-5s", "ml-x-3", "ml-x-3s", "ml-x-4", "ml-x-4s", "ml-x-5", "ml-x-5s", "mn-x-3", "mn-x-3s", "mn-x-4", "mn-x-4s", "mn-x-5", "mn-x-5s", "mni-x-3", "mni-x-3s", "mni-x-4", "mni-x-4s", "mni-x-5", "mni-x-5s", "mr-x-3", "mr-x-3s", "mr-x-4", "mr-x-4s", "mr-x-5", "mr-x-5s", "ms-x-3", "ms-x-3s", "ms-x-4", "ms-x-4s", "ms-x-5", "ms-x-5s", "my-x-3", "my-x-3s", "my-x-4", "my-x-4s", "my-x-5", "my-x-5s", "nb-x-3", "nb-x-3s", "nb-x-4", "nb-x-4s", "nb-x-5", "nb-x-5s", "ne-x-3", "ne-x-3s", "ne-x-4", "ne-x-4s", "ne-x-5", "ne-x-5s", "nl-x-3", "nl-x-3s", "nl-x-4", "nl-x-4s", "nl-x-5", "nl-x-5s", "nn-x-3", "nn-x-3s", "nn-x-4", "nn-x-4s", "nn-x-5", "nn-x-5s", "no-x-3", "no-x-3s", "no-x-4", "no-x-4s", "no-x-5", "no-x-5s", "or-x-3", "or-x-3s", "or-x-4", "or-x-4s", "or-x-5", "or-x-5s", "pa-x-3", "pa-x-3s", "pa-x-4", "pa-x-4s", "pa-x-5", "pa-x-5s", "pcm-x-3", "pcm-x-3s", "pcm-x-4", "pcm-x-4s", "pcm-x-5", "pcm-x-5s", "pl-x-3", "pl-x-3s", "pl-x-4", "pl-x-4s", "pl-x-5", "pl-x-5s", "ps-x-3", "ps-x-3s", "ps-x-4", "ps-x-5", "ps-x-5s", "pt-x-3", "pt-x-3s", "pt-x-4", "pt-x-4s", "pt-x-5", "pt-x-5s", "qu-x-3", "qu-x-3s", "qu-x-5", "qu-x-5s", "raj-x-3", "raj-x-3s", "raj-x-5", "raj-x-5s", "rm-x-3", "rm-x-3s", "rm-x-4", "rm-x-4s", "rm-x-5", "rm-x-5s", "ro-x-3", "ro-x-3s", "ro-x-4", "ro-x-4s", "ro-x-5", "ro-x-5s", "ru-x-3", "ru-x-3s", "ru-x-4", "ru-x-4s", "ru-x-5", "ru-x-5s", "sa-x-3", "sa-x-3s", "sa-x-4", "sa-x-5", "sa-x-5s", "sat-x-3", "sat-x-3s", "sat-x-4", "sat-x-4s", "sat-x-5", "sat-x-5s", "sc-x-3", "sc-x-3s", "sc-x-4", "sc-x-4s", "sc-x-5", "sc-x-5s", "sd-Deva-x-3", "sd-Deva-x-3s", "sd-Deva-x-4", "sd-Deva-x-4s", "sd-Deva-x-5", "sd-Deva-x-5s", "sd-x-3", "sd-x-3s", "sd-x-4", "sd-x-4s", "sd-x-5", "sd-x-5s", "si-x-3", "si-x-3s", "si-x-4", "si-x-4s", "si-x-5", "si-x-5s", "sk-x-3", "sk-x-3s", "sk-x-4", "sk-x-4s", "sk-x-5", "sk-x-5s", "sl-x-3", "sl-x-3s", "sl-x-4", "sl-x-4s", "sl-x-5", "sl-x-5s", "so-x-3", "so-x-3s", "so-x-4", "so-x-4s", "so-x-5", "so-x-5s", "sq-x-3", "sq-x-3s", "sq-x-4", "sq-x-4s", "sq-x-5", "sq-x-5s", "sr-Latn-XK-x-3", "sr-Latn-XK-x-3s", "sr-Latn-x-3", "sr-Latn-x-3s", "sr-Latn-x-4", "sr-Latn-x-4s", "sr-Latn-x-5", "sr-Latn-x-5s", "sr-ME-x-3", "sr-ME-x-3s", "sr-XK-x-3", "sr-XK-x-3s", "sr-x-3", "sr-x-3s", "sr-x-4", "sr-x-4s", "sr-x-5", "sr-x-5s", "su-x-3", "su-x-3s", "su-x-4", "su-x-4s", "su-x-5", "su-x-5s", "sv-x-3", "sv-x-3s", "sv-x-4", "sv-x-4s", "sv-x-5", "sv-x-5s", "sw-x-3", "sw-x-3s", "sw-x-4", "sw-x-4s", "sw-x-5", "sw-x-5s", "ta-x-3", "ta-x-3s", "ta-x-4", "ta-x-4s", "ta-x-5", "ta-x-5s", "te-x-3", "te-x-3s", "te-x-4", "te-x-4s", "te-x-5", "te-x-5s", "tg-x-3", "tg-x-3s", "tg-x-4", "tg-x-4s", "tg-x-5", "tg-x-5s", "th-x-3", "th-x-3s", "th-x-4", "th-x-4s", "th-x-5", "th-x-5s", "ti-x-3", "ti-x-3s", "ti-x-4", "ti-x-4s", "ti-x-5", "ti-x-5s", "tk-x-3", "tk-x-3s", "tk-x-4", "tk-x-4s", "tk-x-5", "tk-x-5s", "to-x-3", "to-x-3s", "to-x-4", "to-x-4s", "to-x-5", "to-x-5s", "tr-x-3", "tr-x-3s", "tr-x-4", "tr-x-4s", "tr-x-5", "tr-x-5s", "tt-x-3", "tt-x-3s", "tt-x-5", "tt-x-5s", "uk-x-3", "uk-x-3s", "uk-x-4", "uk-x-4s", "uk-x-5", "uk-x-5s", "und-x-3", "und-x-3s", "und-x-4", "und-x-4s", "und-x-5", "und-x-5s", "ur-x-3", "ur-x-3s", "ur-x-4", "ur-x-4s", "ur-x-5", "ur-x-5s", "uz-Cyrl-x-3", "uz-Cyrl-x-3s", "uz-Cyrl-x-4", "uz-Cyrl-x-4s", "uz-Cyrl-x-5", "uz-Cyrl-x-5s", "uz-x-3", "uz-x-3s", "uz-x-4", "uz-x-4s", "uz-x-5", "uz-x-5s", "vi-x-3", "vi-x-3s", "vi-x-5", "vi-x-5s", "wo-x-3", "wo-x-3s", "wo-x-5", "wo-x-5s", "xh-x-3", "xh-x-3s", "xh-x-5", "xh-x-5s", "yo-BJ-x-3", "yo-BJ-x-3s", "yo-BJ-x-4", "yo-BJ-x-4s", "yo-BJ-x-5", "yo-BJ-x-5s", "yo-x-3", "yo-x-3s", "yo-x-4", "yo-x-4s", "yo-x-5", "yo-x-5s", "yrl-x-3", "yrl-x-3s", "yrl-x-4", "yrl-x-4s", "yrl-x-5", "yrl-x-5s", "yue-Hans-x-3", "yue-Hans-x-3s", "yue-Hans-x-5", "yue-Hans-x-5s", "yue-x-3", "yue-x-3s", "yue-x-5", "yue-x-5s", "zh-Hant-x-3", "zh-Hant-x-3s", "zh-Hant-x-5", "zh-Hant-x-5s", "zh-x-3", "zh-x-3s", "zh-x-5", "zh-x-5s", "zu-x-3", "zu-x-3s", "zu-x-4", "zu-x-4s", "zu-x-5", "zu-x-5s"];
}
